<?php
// $Id: pnuser.php,v 1.1.1.1 2002/09/15 22:26:15 root Exp $
// ----------------------------------------------------------------------
// POST-NUKE Content Management System
// Copyright (C) 2002 by the PostNuke Development Team.
// http://www.postnuke.com/
// ----------------------------------------------------------------------
// Based on:
// PHP-NUKE Web Portal System - http://phpnuke.org/
// Thatware - http://thatware.org/
// ----------------------------------------------------------------------
// LICENSE
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License (GPL)
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WIthOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// To read the license please visit http://www.gnu.org/copyleft/gpl.html
// ----------------------------------------------------------------------
// Original Author of file: Jim McDonald
// Purpose of file:  Template user display functions
// ----------------------------------------------------------------------

/**
 * the main user function
 * This function is the default function, and is called whenever the module is
 * initiated without defining arguments.  As such it can be used for a number
 * of things, but most commonly it either just shows the module menu and
 * returns or calls whatever the module designer feels should be the default
 * function (often this is the view() function)
 */
function template_user_main()
{
    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $output = new pnHTML();

    
    // Security check - important to do this as early as possible to avoid
    // potential security holes or just too much wasted processing.  For the
    // main function we want to check that the user has at least overview
    // privilege for some item within this component, or else they won't be
    // able to see anything and so we refuse access altogether.  The lowest
    // level of access for administration depends on the particular module, but
    // it is generally either 'overview' or 'read'
    if (!pnSecAuthAction(0, 'Template::', '::', ACCESS_READ)) {
        $output->Text(_TEMPLATENOAUTH);
        return $output->GetOutput();
    }

    // Add menu to output - it helps if all of the module pages have a standard
    // menu at their head to aid in navigation
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    $output->Text(template_usermenu());
    $output->SetInputMode(_PNH_PARSEINPUT);

    // Return the output that has been generated by this function
    return $output->GetOutput();
}

/**
 * view items
 * This is a standard function to provide an overview of all of the items
 * available from the module.
 */
function template_user_view()
{
    // Get parameters from whatever input we need.  All arguments to this
    // function should be obtained from pnVarCleanFromInput(), getting them
    // from other places such as the environment is not allowed, as that makes
    // assumptions that will not hold in future versions of PostNuke
    $startnum = pnVarCleanFromInput('startnum');

    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $output = new pnHTML();

    // Add menu to output - it helps if all of the module pages have a standard
    // menu at their head to aid in navigation
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    $output->Text(template_usermenu());
    $output->SetInputMode(_PNH_PARSEINPUT);

    // Security check - important to do this as early as possible to avoid
    // potential security holes or just too much wasted processing
    if (!pnSecAuthAction(0, 'Templates::', '::', ACCESS_OVERVIEW)) {
        $output->Text(_TEMPLATENOAUTH);
        return $output->GetOutput();
    }

    // Load API.  All of the actual work for obtaining information on the items
    // is done within the API, so we need to load that in before we can do
    // anything.  If the API fails to load an appropriate error message is
    // posted and the function returns
    if (!pnModAPILoad('Template', 'user')) {
        $output->Text(_LOADFAILED);
        return $output->GetOutput();
    }

    // The API function is called.  The arguments to the function are passed in
    // as their own arguments array
    $items = pnModAPIFunc('Template',
                          'user',
                          'getall',
                          array('startnum' => $startnum,
                                'numitems' => pnModGetVar('Template',
                                                          'itemsperpage')));

    // The return value of the function is checked here, and if the function
    // suceeded then an appropriate message is posted.  Note that if the
    // function did not succeed then the API function should have already
    // posted a failure message so no action is required
    if ($items == false) {
        $output->Text(_TEMPLATEITEMFAILED);
    }

    // Loop through each item and display it.  Note the use of pnVarCensor() to
    // remove any words from the name that the administrator has deemed
    // unsuitable for the site
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    foreach ($items as $item) {
        
        if (pnSecAuthAction(0,
                            'Templates::',
                            "$item[name]::$item[tid]",
                            ACCESS_READ)) {
            $output->URL(pnModURL('Template',
                                  'user',
                                  'display',
                                  array('tid' => $item['tid'])),
                         pnVarPrepForDisplay(pnVarCensor($item['name'])));
            $output->Linebreak();
        }
    }
    $output->SetInputMode(_PNH_PARSEINPUT);

    // Call the pnHTML helper function to produce a pager in case of there
    // being many items to display.
    //
    // Note that this function includes another user API function.  The
    // function returns a simple count of the total number of items in the item
    // table so that the pager function can do its job properly
    $output->Pager($startnum,
            pnModAPIFunc('Template', 'user', 'countitems'),
                         pnModURL('Template',
                                  'user',
                                  'view',
                                  array('startnum' => '%%')),
                         pnModGetVar('Template', 'itemsperpage'));

    // Return the output that has been generated by this function
    return $output->GetOutput();
}

/**
 * display item
 * This is a standard function to provide detailed informtion on a single item
 * available from the module.
 */
function template_user_display($args)
{
    // Get parameters from whatever input we need.  All arguments to this
    // function should be obtained from pnVarCleanFromInput(), getting them
    // from other places such as the environment is not allowed, as that makes
    // assumptions that will not hold in future versions of PostNuke.
    list($tid,
         $objectid) = pnVarCleanFromInput('tid',
                                          'objectid');

    // User functions of this type can be called by other modules.  If this
    // happens then the calling module will be able to pass in arguments to
    // this function through the $args parameter.  Hence we extract these
    // arguments *after* we have obtained any form-based input through
    // pnVarCleanFromInput().
    extract($args);

    // At this stage we check to see if we have been passed $objectid, the
    // generic item identifier.  This could have been passed in by a hook or
    // through some other function calling this as part of a larger module, but
    // if it exists it overrides $tid
    //
    // Note that this module couuld just use $objectid everywhere to avoid all
    // of this munging of variables, but then the resultant code is less
    // descriptive, especially where multiple objects are being used.  The
    // decision of which of these ways to go is up to the module developer
    if (!empty($objectid)) {
        $tid = $objectid;
    }

    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $output = new pnHTML();

    // Add menu to output - it helps if all of the module pages have a standard
    // menu at their head to aid in navigation
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    $output->Text(template_usermenu());
    $output->SetInputMode(_PNH_PARSEINPUT);

    // Load API.  All of the actual work for obtaining information on the items
    // is done within the API, so we need to load that in before we can do
    // anything.  If the API fails to load an appropriate error message is
    // posted and the function returns
    if (!pnModAPILoad('Template', 'user')) {
        $output->Text(_LOADFAILED);
        return $output->GetOutput();
    }

    // The API function is called.  The arguments to the function are passed in
    // as their own arguments array
    $item = pnModAPIFunc('Template',
                          'user',
                          'get',
                          array('tid' => $tid));

    // The return value of the function is checked here, and if the function
    // suceeded then an appropriate message is posted.  Note that if the
    // function did not succeed then the API function should have already
    // posted a failure message so no action is required
    if ($item == false) {
        $output->Text(_TEMPLATEITEMFAILED);
        return $output->GetOutput();
    }

    // Display details of the item.  Note the use of pnVarCensor() to remove
    // any words from the name that the administrator has deemed unsuitable for
    // the site.  Also note that a module variable is used here to determine
    // whether not not parts of the item information should be displayed in
    // bold type or not
    $output->Text(_TEMPLATENAME . ': ');
    if (pnModGetVar('Template', 'bold')) {
        $output->BoldText(pnVarCensor($item['name']));
    } else {
        $output->Text(pnVarCensor($item['name']));
    }
    $output->Linebreak(2);
    $output->Text(_TEMPLATENUMBER . ': ');
    $output->Text($item['number']);
    $output->Linebreak(2);


    // Let any hooks know that we are displaying an item.  As this is a display
    // hook we're passing a URL as the extra info, which is the URL that any
    // hooks will show after they have finished their own work.  It is normal
    // for that URL to bring the user back to this function
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    $output->Text(pnModCallHooks('item',
                                 'display',
                                 $tid,
                                 pnModURL('Template',
                                          'user',
                                          'display',
                                          array('tid' => $tid))));
    $output->SetInputMode(_PNH_PARSEINPUT);

    // Return the output that has been generated by this function
    return $output->GetOutput();
}

/**
 * generate menu fragment
 */
function template_usermenu()
{
    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $output = new pnHTML();

    // Display status message if any.  Note that in future this functionality
    // will probably be in the theme rather than in this menu, but this is the
    // best place to keep it for now

    // Start options menu
    $output->Text(pnGetStatusMsg());
    $output->Linebreak(2);

    // Menu options.  These options are all added in a single row, to add
    // multiple rows of options the code below would just be repeated
    $output->TableStart(_TEMPLATE);
    $output->SetOutputMode(_PNH_RETURNOUTPUT);
    $columns = array();
    $columns[] = $output->URL(pnModURL('Template',
                                       'user',
                                       'view'),
                              _TEMPLATEVIEW);
    $output->SetOutputMode(_PNH_KEEPOUTPUT);
    $output->SetInputMode(_PNH_VERBATIMINPUT);
    $output->TableAddRow($columns);
    $output->SetInputMode(_PNH_PARSEINPUT);
    $output->TableEnd();

    // Return the output that has been generated by this function
    return $output->GetOutput();
}

?>